<?php
// app/Http/Controllers/PrediksiController.php

namespace App\Http\Controllers;

use App\Models\Prediksi;
use App\Models\Desa;
use App\Services\AIPredictionService;
use Illuminate\Http\Request;
use Carbon\Carbon;

class PrediksiController extends Controller
{
    protected $aiService;
    
    public function __construct(AIPredictionService $aiService)
    {
        $this->middleware('auth');
        $this->aiService = $aiService;
    }
    
    /**
     * Display prediksi page
     */
    public function index()
    {
        $user = auth()->user();
        
        // Query prediksi
        $prediksiQuery = Prediksi::with('desa', 'dusun')
            ->where('tanggal_prediksi', '>=', now()->format('Y-m-d'))
            ->orderBy('tanggal_prediksi', 'asc');
        
        // Filter by user role
        if ($user->role === 'warga' && $user->desa_id) {
            $prediksiQuery->where('desa_id', $user->desa_id);
        }
        
        $prediksi = $prediksiQuery->paginate(20);
        
        // Statistik
        $stats = [
            'total_kritis' => Prediksi::where('tingkat_risiko', 'kritis')
                ->where('tanggal_prediksi', now()->format('Y-m-d'))
                ->count(),
            'total_waspada' => Prediksi::where('tingkat_risiko', 'waspada')
                ->where('tanggal_prediksi', now()->format('Y-m-d'))
                ->count(),
            'total_aman' => Prediksi::where('tingkat_risiko', 'aman')
                ->where('tanggal_prediksi', now()->format('Y-m-d'))
                ->count(),
        ];
        
        return view('dashboard.prediksi', compact('prediksi', 'stats'));
    }
    
    /**
     * Generate prediksi untuk desa tertentu
     */
    public function generate(Request $request)
    {
        $validated = $request->validate([
            'desa_id' => 'required|exists:desa,id',
            'days' => 'nullable|integer|min:1|max:14',
        ]);
        
        $desaId = $validated['desa_id'];
        $days = $validated['days'] ?? 7;
        
        // Generate prediksi menggunakan AI
        $predictions = $this->aiService->predictDroughtRisk($desaId, $days);
        
        return redirect()->back()->with('success', 
            "Prediksi {$days} hari untuk desa berhasil di-generate!");
    }
    
    /**
     * Generate prediksi untuk semua desa
     */
    public function generateAll()
    {
        if (!auth()->user()->isAdmin()) {
            return redirect()->back()->with('error', 'Hanya admin yang bisa generate semua desa.');
        }
        
        $allDesa = Desa::all();
        $totalGenerated = 0;
        
        foreach ($allDesa as $desa) {
            $this->aiService->predictDroughtRisk($desa->id, 7);
            $totalGenerated++;
        }
        
        return redirect()->back()->with('success', 
            "Berhasil generate prediksi untuk {$totalGenerated} desa!");
    }
    
    /**
     * Show detail prediksi
     */
    public function show($id)
    {
        $prediksi = Prediksi::with('desa', 'dusun')->findOrFail($id);
        
        return view('dashboard.prediksi-detail', compact('prediksi'));
    }
    
    /**
     * Get prediksi data for chart (AJAX)
     */
    public function chartData(Request $request)
    {
        $desaId = $request->get('desa_id');
        $days = $request->get('days', 7);
        
        $prediksi = Prediksi::where('desa_id', $desaId)
            ->where('tanggal_prediksi', '>=', now()->format('Y-m-d'))
            ->orderBy('tanggal_prediksi', 'asc')
            ->take($days)
            ->get();
        
        $chartData = [
            'labels' => $prediksi->pluck('tanggal_prediksi')->map(function($date) {
                return Carbon::parse($date)->format('d M');
            }),
            'curah_hujan' => $prediksi->pluck('curah_hujan'),
            'kelembapan' => $prediksi->pluck('kelembapan_tanah'),
            'risiko_score' => $prediksi->pluck('risiko_score'),
        ];
        
        return response()->json($chartData);
    }
    
    /**
     * Tampilkan peta dengan marker desa
     */
    public function peta()
    {
        $user = auth()->user();
        
        // Get desa with latest prediction
        $desaQuery = Desa::with(['prediksi' => function($query) {
            $query->where('tanggal_prediksi', now()->format('Y-m-d'))
                  ->orderBy('risiko_score', 'desc')
                  ->limit(1);
        }]);
        
        if ($user->role === 'warga' && $user->desa_id) {
            $desaQuery->where('id', $user->desa_id);
        }
        
        $desa = $desaQuery->get();
        
        return view('dashboard.peta', compact('desa'));
    }
}
